/**
 * \file: mspin_connection_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin ConnectionAdapter
 *
 * \component: MSPIN
 *
 * \author: Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * \history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/

#include "mspin_connection_adapter.h"
#include "mspin_connection_tcp_server.h"
#include "mspin_connection_tcp_client.h"
#include "mspin_connection_tcp_listener.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_udp_broadcast.h"
#include "mspin_udp_unicast.h"
#include "mspin_logging.h"
#include "mspin_core_adapter.h"
#include "mspin_connection_adapter_iap2.h"
#include "mspin_connection_tcp.h"
#include "mspin_measurement.h"
#include <aoap.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>  // sleep()
#include <string.h>  // strcpy()
#include <pthread.h>

#ifndef MSPIN_ENABLE_TCP_IP_CLIENT
#define MSPIN_DISABLE_TCP_IP_CLIENT
#endif //MSPIN_ENABLE_TCP_IP_CLIENT

//#define MSPINUSETCP  //uncomment to test via TCP/CommandlinePhone
#ifndef MSPINUSETCP

//ToDo: Add a configuration parameter for this
// limit to 16kB (used for LCN2EVO)
#define MSPIN_AOAP_MAX_READ_LIMIT (16*1024)   //128*1024
#define MSPIN_AOAP_DEFAULT_WRITE_TIMEOUT 3000 //3 seconds
#define MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT 1000
#define MSPIN_DEFAULT_READ_TIMEOUT 1000 //1 second

U32 gLastFrameSize = 0;
static U32 gSizeSummingUp = 0;
static U32 gLastPacketSize = 0;

static S32 gAccessoryId = -1;
static BOOL gEnablePerformanceMeasurements = FALSE;

static pthread_mutex_t gCommCountMutex;

mspin_connectionHandle_t* mspin_conn_createHandle(void)
{
    mspin_connectionHandle_t* pConnectionHandle = (mspin_connectionHandle_t*)malloc(sizeof(mspin_connectionHandle_t));
    if (pConnectionHandle)
    {
        memset(pConnectionHandle, 0, sizeof(mspin_connectionHandle_t)); //sets also the pointers to NULL

        //Initialize only values which aren't 0
        pConnectionHandle->result = MSPIN_SUCCESS;
        pConnectionHandle->accessoryId = -1;
        pConnectionHandle->deviceId = -1;
        pConnectionHandle->port = -1;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s() FATAL ERROR: Failed to allocate connection handle",
                __FUNCTION__);
    }

    return pConnectionHandle;
}

void mspin_conn_releaseHandle(mspin_connectionHandle_t **ppConnectionHandle)
{
    mspin_connectionHandle_t* pConnectionHandle = *ppConnectionHandle;

    if (pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(connHandle=%p) Deleting connHandle",
                __FUNCTION__, pConnectionHandle);

        if (pConnectionHandle->pReadDevice)
        {
            free(pConnectionHandle->pReadDevice);
            pConnectionHandle->pReadDevice = NULL;
        }

        if (pConnectionHandle->pWriteDevice)
        {
            free(pConnectionHandle->pWriteDevice);
            pConnectionHandle->pWriteDevice = NULL;
        }

        free(pConnectionHandle);
        *ppConnectionHandle = NULL;
    }
}

static void mspin_conn_connectAOAP_CB(int accessoryId, int deviceId, int result, void *token, unsigned int audioSupport)
{
    mspin_context_t* pContext = (mspin_context_t*)token;
    if (pContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(accId=%d,devId=%d,result=%d, ctx=%p, audio='%s'){%d} -> unlocking",
                __FUNCTION__, accessoryId, deviceId, result, pContext,
                (0 == audioSupport) ? "disabled" : "enabled", pContext->instanceId);

        mspin_connectionHandle_t* pConnectionHandle = mspin_conn_createHandle();
        if (pConnectionHandle)
        {
            pConnectionHandle->accessoryId = accessoryId;
            pConnectionHandle->deviceId = deviceId;
            pConnectionHandle->result = result;
            pConnectionHandle->audioSupport = audioSupport;
            pConnectionHandle->connectionType = MSPIN_CONNECTION_AOAP;
        }
        pContext->pConnectionHandle = pConnectionHandle;
        sem_post(&(pContext->connectLock));
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(accId=%d,devId=%d,result=%d, ctx=%p, audio='%s'){} FATAL ERROR: token is NULL",
                __FUNCTION__, accessoryId, deviceId, result, pContext,
                (0 == audioSupport) ? "disabled" : "enabled");
    }
}

void mspin_conn_enableAOAPPerfomanceMeasurement(void)
{
    gEnablePerformanceMeasurements = TRUE;
}

mspin_connectionHandle_t* mspin_conn_connectAOAP(mspin_context_t* pContext,
        U32 vendorId, U32 productId, const char* serial, bool* pAudioSupport)
{
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(%#x,%#x,%s,audio %s){} FATAL ERROR: pContext is NULL",
                __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled");
        return NULL;
    }

    // connect
    mspin_log_printLn(eMspinVerbosityInfo, "%s(%#x,%#x,%s,audio %s){%d}",
            __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled",
            pContext->instanceId);

    mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

    if (gAccessoryId < 0)
    {
        if (mspin_log_getDestination() & MSPIN_LOGGING_TO_FILE)
        {
            aoap_set_log_destination(eLogToFile, true, NULL);
        }
        else if (mspin_log_getDestination() & MSPIN_LOGGING_TO_STDOUT)
        {
            aoap_set_log_destination(eLogToStdout, true, NULL);
        }

        if (eMspinVerbosityDefault != mspin_log_getVerbosityLevel()) //set only if not using default
        {
            aoap_set_log_level((int)mspin_log_getVerbosityLevel(), mspin_log_getTimestamps());
        }

        if (gEnablePerformanceMeasurements)
        {
            aoap_enable_performance_measurement();
        }

        t_aoap_accessory_param accessory;
    	accessory.manufacturer = pContext->AccVendor;
        accessory.modelName = pContext->AccModel;
        accessory.description = pContext->AccDescription;
        accessory.version = pContext->AccVersion;
        accessory.uri = pContext->AccUri;
        accessory.serial = pContext->AccSerial;
        accessory.enableAudio = 1; //enable audio for accessory in general
        gAccessoryId = aoap_create_accessory(&accessory);
    }

    (void)sem_init(&(pContext->connectLock), 0, 0);  // binary semaphore locked

    // call AOAP
    int rc = aoap_connect(gAccessoryId, vendorId, productId, serial, &mspin_conn_connectAOAP_CB,
            *pAudioSupport, pContext);

    if ((rc < 0) && (AOAP_ERROR_ALREADY_DONE != rc)) //in case of failure other than already in accessory mode
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(%#x,%#x,%s,audio %s){%d} ERROR: aoap_connect() failed: %d",
                __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled",
                pContext->instanceId, rc);
    }
    else
    {
        if (AOAP_ERROR_ALREADY_DONE == rc)
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s(%#x,%#x,%s,audio %s){%d} WARNING: Device is already in accessory mode => waiting for connect ...",
                    __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled",
                    pContext->instanceId, rc);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(%#x,%#x,%s,audio %s){%d} waiting for connect ...",
                    __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled", pContext->instanceId);
        }

        (void)sem_wait(&(pContext->connectLock));  // wait until AOAP has connected the phone (see callback 'mspin_conn_connectAOAP_CB')

        //Check result
        if (pContext->pConnectionHandle)
        {
            if (pContext->pConnectionHandle->result == AOAP_SUCCESS)
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(%#x,%#x,%s,audio %s){%d} result=%d",
                        __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled", pContext->instanceId,
                        pContext->pConnectionHandle->result);
            }
            else if (pContext->pConnectionHandle->result == AOAP_ERROR_ALREADY_DONE)
            {
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s(%#x,%#x,%s,audio %s){%d} reconnected to device in accessory mode (%d)",
                        __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled", pContext->instanceId,
                        pContext->pConnectionHandle->result);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(%#x,%#x,%s,audio %s){%d} ERROR: An error=%d occurred => release connection handle and return NULL",
                        __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled", pContext->instanceId,
                        pContext->pConnectionHandle->result);

                mspin_conn_releaseHandle(&(pContext->pConnectionHandle));
                pContext->pConnectionHandle = NULL;
            }
        }
        else
        {
            //Error: No connection handle set => return NULL
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(%#x,%#x,%s,audio %s){%d} ERROR: No connection handle received",
                    __FUNCTION__, vendorId, productId, serial, *pAudioSupport ? "enabled" : "disabled", pContext->instanceId);
        }
    }

    return pContext->pConnectionHandle;
}


void mspin_conn_disconnectAoap(mspin_context_t* pContext)
{
    if (pContext)
    {
        if (pContext->pConnectionHandle)
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s aoap_defer_delete_accessory(%d,%d)", __FUNCTION__,
                    pContext->pConnectionHandle->accessoryId, pContext->pConnectionHandle->deviceId);
            aoap_defer_delete_accessory(pContext->pConnectionHandle->accessoryId, pContext->pConnectionHandle->deviceId);
            gAccessoryId = -1;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s pConnectionHandle is NULL", __FUNCTION__);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s FATAL ERROR: pContext is NULL", __FUNCTION__);
    }
}


mspin_connectionHandle_t* mspin_conn_connectiAP2(mspin_context_t* pContext,
        U32 vendorId, U32 productId, const char* serial, const char *pWriteDevice,
        const char *pReadDevice, bool hostMode)
{
    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pContext=%p, %.4x:%.4x, s/n='%s'){} FATAL ERROR: pContext is NULL",
                __FUNCTION__, pContext, vendorId, productId, serial);
        return NULL;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(pContext=%p, %.4x:%.4x, s/n='%s', writeDevice='%s', readDevice='%s', %sMode){%d}",
            __FUNCTION__, pContext, vendorId, productId, serial,
            pWriteDevice, pReadDevice, hostMode ? "host" : "device",
            pContext->instanceId);

    //Make sure connection handle is freed
    mspin_conn_releaseHandle(&(pContext->pConnectionHandle));

    pContext->pConnectionHandle = mspin_conn_createHandle();
    if (pContext->pConnectionHandle)
    {
        pContext->pConnectionHandle->result = MSPIN_SUCCESS;
        if (hostMode)
        {
            pContext->pConnectionHandle->connectionType = MSPIN_CONNECTION_IAP_HOST;
        }
        else
        {
            pContext->pConnectionHandle->connectionType = MSPIN_CONNECTION_IAP_DEVICE;
        }

        pContext->pConnectionHandle->pReadDevice = (char*)malloc(strlen(pReadDevice)+1);
        strcpy(pContext->pConnectionHandle->pReadDevice, pReadDevice);

        pContext->pConnectionHandle->pWriteDevice = (char*)malloc(strlen(pWriteDevice)+1);
        strcpy(pContext->pConnectionHandle->pWriteDevice, pWriteDevice);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pContext=%p, %.4x:%.4x, s/n='%s', writeDevice='%s', readDevice='%s', %sMode){%d} FATAL ERROR: Failed to allocate memory for connection handle",
                __FUNCTION__, pContext, vendorId, productId, serial,
                pWriteDevice, pReadDevice, hostMode ? "host" : "device",
                pContext->instanceId);
    }

    return pContext->pConnectionHandle;
}


void mspin_conn_disconnectiAP2(void)
{
    mspin_conn_iap2_shutdownConnection();
}

size_t mspin_conn_receive(void* context, UInt8* buffer, size_t bufferSize, void* pMspinContext)
{
    S32 bytesRead = 0;
    size_t maxReadSize = MSPIN_AOAP_MAX_READ_LIMIT;
    mspin_context_t* pContext = (mspin_context_t*)pMspinContext;

    MSPIN_UNUSED(context);

    if (pContext && pContext->pConnectionHandle && !pContext->pConnectionHandle->connectionError)
    {
        mspin_conn_incrementCommCount(pContext);
        //mspin_dbgPrintLine(eMspinVerbosityVerbose, "%s() -> entry", __FUNCTION__);

        switch (pContext->pConnectionHandle->connectionType)
        {
            case MSPIN_CONNECTION_AOAP:
            {
                //Limit buffer to 128 kByte if requested buffer is bigger in order to improve
                // performance because libusb seems to have problems when a big buffer is used
                if (bufferSize < MSPIN_AOAP_MAX_READ_LIMIT)
                {
                    maxReadSize = bufferSize;
                }

                bytesRead = aoap_read(pContext->pConnectionHandle->accessoryId,
                     pContext->pConnectionHandle->deviceId, (unsigned char *)buffer,
                     maxReadSize, MSPIN_DEFAULT_READ_TIMEOUT);

                mspin_log_printLn((gSizeSummingUp==0) ? eMspinVerbosityDebug : eMspinVerbosityVerbose,
                     "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received using AOAP",
                     __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                     bytesRead);

                gSizeSummingUp += bytesRead;
                if (bytesRead < MSPIN_AOAP_MAX_READ_LIMIT)
                {
                    gLastFrameSize = gSizeSummingUp;   // frame complete but it can include other data
                    gSizeSummingUp = 0;                // reset for next frame
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(buf=%p, size=%d, conn=%d){%d} last %d of %d bytes received",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId, bytesRead, gLastFrameSize);
                }
                break;
            }
            case MSPIN_CONNECTION_IAP_HOST:
            {
                bytesRead = mspin_conn_iap2_receive((U8*)buffer, (U32)bufferSize, MSPIN_DEFAULT_READ_TIMEOUT);
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received using iAP2 (host mode)",
                        __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                        bytesRead);
                break;
            }
            case MSPIN_CONNECTION_IAP_DEVICE:
            {
                if (pContext->pConnectionHandle->eapReceive_CB)
                {
                    bytesRead = pContext->pConnectionHandle->eapReceive_CB((U8*)buffer, (U32)bufferSize, pContext->pConnectionHandle->eapContext);
                    if (bytesRead == 0)
                    {
                        mspin_log_printLn(eMspinVerbosityVerbose,
                            "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received (=timeout) using iAP1/2 (device mode)",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                            bytesRead);
                    }
                    else if (bytesRead < 0)
                    {
                        mspin_log_printLn(eMspinVerbosityError,
                            "%s(buf=%p, size=%d, conn=%d){%d} ERROR: Failed to receive bytes with %d using iAP1/2 (device mode)",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                            bytesRead);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received using iAP1/2 (device mode)",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                            bytesRead);
                    }
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(buf=%p, size=%d, conn=%d){%d} ERROR: Not in host mode and no callback set",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId);
                }
                break;
            }
            case MSPIN_CONNECTION_CLIENT_TCP:
            case MSPIN_CONNECTION_SERVER_TCP:
            {
                bytesRead = mspin_tcp_receive(pContext->pConnectionHandle, (U8*)buffer, (U32)bufferSize, MSPIN_DEFAULT_READ_TIMEOUT);
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received using TCP/IP",
                        __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                        bytesRead);

                //Ignore too small packets
                if (bytesRead > 100)
                {
                    gLastFrameSize += bytesRead;
                }
                break;
            }
            case MSPIN_CONNECTION_CLIENT_TLS:
            case MSPIN_CONNECTION_SERVER_TLS:
            {
                bytesRead = mspin_tls_receive(pContext->pConnectionHandle, (U8*)buffer, (U32)bufferSize, MSPIN_DEFAULT_READ_TIMEOUT);
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(buf=%p, size=%d, conn=%d){%d} %d bytes received using TLS over TCP/IP",
                        __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId,
                        bytesRead);

                //Ignore too small packets
                if (bytesRead > 100)
                {
                    gLastFrameSize += bytesRead;
                }
                break;
            }
            default:
            {
                break;
            }
        }

        //Counting frame size in iAP case //ToDo: Why only in iAP cases?
        if ((MSPIN_CONNECTION_IAP_HOST == pContext->pConnectionHandle->connectionType)
                || (MSPIN_CONNECTION_IAP_DEVICE == pContext->pConnectionHandle->connectionType))
        {
            if (bytesRead > 50) //ignore packets which are too small like pings...
            {
                gSizeSummingUp += bytesRead;

                //This code uses the fact that the packet size will not be less the previous packet except we reached
                // the frame end
                if (bytesRead < (S32)gLastPacketSize)
                {
                    gLastFrameSize = gSizeSummingUp;   // packet complete
                    gSizeSummingUp = 0; // reset for next packet
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(buf=%p, size=%d, conn=%d){%d} frame with est. %d bytes received",
                            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId, gLastFrameSize);
                }

                gLastPacketSize = bytesRead;
            }
        }

        // Core only interprets -1 as EOF:
        if (bytesRead < 0)
        {
            bytesRead = -1;
            if (pContext->pConnectionHandle)   // pConnectionHandle might have been reset meanwhile
            {
                pContext->pConnectionHandle->connectionError = TRUE;
            }
        }
        else
        {
            mspin_measure_setDataReceived(buffer, bytesRead);
        }

        mspin_conn_decrementCommCount(pContext);
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
            "%s(buf=%p, size=%d, conn=%d){-} FATAL ERROR: Context is NULL",
            __FUNCTION__, buffer, bufferSize, pMspinContext);
        bytesRead = -1;
    }
    else if (!pContext->pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
            "%s(buf=%p, size=%d, conn=%d){%d} FATAL ERROR: Connection handle is NULL",
            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId);
        bytesRead = -1;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
            "%s(buf=%p, size=%d, conn=%d){%d} ERROR: connection broken -> cancel reads",
            __FUNCTION__, buffer, bufferSize, pMspinContext, pContext->instanceId);
        bytesRead = -1;
    }
    /* PRQA: Lint Message 571: Agreed with mySPIN core developers. Not an issue. */
    /*lint -save -e571*/
    return (size_t)bytesRead;
    /*lint -restore*/
}


void mspin_conn_aoap_send(mspin_context_t *pContext, UInt8* buffer, size_t bufferSize, void* pMspinContext)
{
    int bytesWritten = 0;

    //During protocol setup the timeout needs to be higher than the default timeout in order to allow a
    // delayed start of the launcher application. But the timeout should not be higher than the protocol
    // initialization timeout. Therefore set it to this timeout
    unsigned int timeout = MSPIN_AOAP_DEFAULT_WRITE_TIMEOUT;

    if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p, buffer=%p, size=%d, connHandle=%p){} FATAL ERROR: pContext is NULL",
                __FUNCTION__, pContext, buffer, bufferSize, pMspinContext);
        return;
    }

    if (!pContext->isReady)
    {
        timeout = pContext->initializationTimeout * 1000;
    }

    mspin_log_printLn(eMspinVerbosityDebug,
            "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} sending bytes using AOAP pthread=%lu",
            __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId,
            (unsigned long int)pthread_self());

    while (pContext->pConnectionHandle && !pContext->pConnectionHandle->connectionError && (bytesWritten < (int)bufferSize))
    {
        bytesWritten = aoap_write(pContext->pConnectionHandle->accessoryId,
                pContext->pConnectionHandle->deviceId, (unsigned char *)(buffer + bytesWritten),
                bufferSize - bytesWritten,
                (MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT < timeout) ? MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT : timeout);

        if (bytesWritten < 0)
        {
            if (AOAP_ERROR_TIMEOUT == bytesWritten)
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} ERROR: AOAP TIMEOUT should not happen with current implementation",
                        __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityWarn,
                        "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} AOAP ERROR %d",
                        __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId, bytesWritten);
            }

            //In case of connection error set connectionError and leave loop (by setting connectionError)
            if (pContext->pConnectionHandle) //check because pConnectionHandle might have been reset meanwhile
            {
                pContext->pConnectionHandle->connectionError = TRUE;
            }
        }
        else if ((0 == bytesWritten) && (pContext->isReady))
        {
            //When no data is written after initialization within a single write operation, we handle it as error to react
            // faster on communication problems. If some data has been written, we will try to write the remaining data
            // within MSPIN_AOAP_DEFAULT_WRITE_TIMEOUT.
            // NOTE: During initialization it can happen, that no data will be written. This happens especially when writing
            // the first time and the user has not given permission on the Android to let the application communicate with
            // the accessory.
            mspin_log_printLn(eMspinVerbosityError,
                    "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} ERROR: No bytes written within %d ms => communication error",
                    __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId,
                    (MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT < timeout) ? MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT : timeout);

            if (pContext->pConnectionHandle) //check because pConnectionHandle might have been reset meanwhile
            {
                pContext->pConnectionHandle->connectionError = TRUE;
            }
        }
        else if (bytesWritten < (int)bufferSize)
        {
            timeout -= (MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT < timeout) ? MSPIN_AOAP_DEFAULT_SINGLE_WRITE_TIMEOUT : timeout; //must match to aoap_write timeout

            if (timeout > 0)
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} Only %d of %d bytes written => try to write remaining bytes in %d ms",
                        __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId, bytesWritten, bufferSize, timeout);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                    "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} ERROR: Timeout. Only %d of %d bytes written => communication error",
                    __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId, bytesWritten, bufferSize);
                
                //In case of timeout reached, leave loop (by setting connectionError)
                if (pContext->pConnectionHandle) //check because pConnectionHandle might have been reset meanwhile
                {
                    pContext->pConnectionHandle->connectionError = TRUE;
                }
            }
        }
        else if (bytesWritten > (int)bufferSize)
        {
            mspin_log_printLn(eMspinVerbosityFatal,
                    "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} FATAL ERROR: More bytes written than buffer size",
                    __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId);

            //Set connection error and leave loop (by setting connectionError)
            if (pContext->pConnectionHandle) //check because pConnectionHandle might have been reset meanwhile
            {
                pContext->pConnectionHandle->connectionError = TRUE;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityVerbose,
                    "%s(context=%p, buffer=%p, size=%d, connHandle=%p){%d} AOAP success",
                    __FUNCTION__, pContext, buffer, bufferSize, pMspinContext, pContext->instanceId);
        }
    }
}


void* mspin_conn_send(void* context, UInt8* buffer, size_t bufferSize, void* pMspinContext)
{
    mspin_context_t* pContext = (mspin_context_t*)pMspinContext;

    MSPIN_UNUSED(context);

    if (pContext && pContext->pConnectionHandle && !pContext->pConnectionHandle->connectionError)
    {
        //If it's a new frame request, reset last frame size
        if ((18 == bufferSize) && buffer && (0x19 == buffer[0]))
        {
            gLastFrameSize = 0;
        }

        mspin_conn_incrementCommCount(pContext);

        switch (pContext->pConnectionHandle->connectionType)
        {
            case MSPIN_CONNECTION_AOAP:
            {
                mspin_conn_aoap_send(pContext, buffer, bufferSize, pMspinContext);
                break;
            }
            case MSPIN_CONNECTION_IAP_HOST:
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} sending bytes using iAP2 (host mode)",
                        __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);

                if (mspin_conn_iap2_send((const U8*)buffer, (U32)bufferSize) < 0)
                {
                    if (pContext->pConnectionHandle)   // pConnectionHandle might have been reset meanwhile
                    {
                        pContext->pConnectionHandle->connectionError = TRUE;
                    }
                }
                break;
            }
            case MSPIN_CONNECTION_IAP_DEVICE:
            {
                if (pContext->pConnectionHandle->eapSend_CB)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} sending bytes using iAP1/2 (device mode)",
                            __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);
                    (void)pContext->pConnectionHandle->eapSend_CB((U8*)buffer, (U32)bufferSize, pContext->pConnectionHandle->eapContext);
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError,
                            "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} ERROR: In device mode but no callback registered",
                            __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);
                }
                break;
            }
            case MSPIN_CONNECTION_CLIENT_TCP:
            case MSPIN_CONNECTION_SERVER_TCP:
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} sending bytes using TCP/IP",
                        __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);

                (void)mspin_tcp_send(pContext->pConnectionHandle, (const U8*)buffer, (U32)bufferSize);
                break;
            }
            case MSPIN_CONNECTION_CLIENT_TLS:
            case MSPIN_CONNECTION_SERVER_TLS:
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} sending bytes using TLS over TCP/IP",
                        __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);
                (void)mspin_tls_send(pContext->pConnectionHandle, (const U8*)buffer, (U32)bufferSize);
                break;
            }
            default:
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){%d} ERROR: Connection type not handled",
                        __FUNCTION__, context, buffer, bufferSize, pMspinContext, pContext->instanceId);
                break;
            }
        }

        mspin_measure_frameRequested((U8*)buffer, bufferSize);
        mspin_conn_decrementCommCount(pContext);
    }
    else if (!pContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){} FATAL ERROR: context is NULL",
                __FUNCTION__, context, buffer, bufferSize, pMspinContext);
    }
    else if (!pContext->pConnectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){} FATAL ERROR: context->pConnectionHandle is NULL",
                __FUNCTION__, context, buffer, bufferSize, pMspinContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(context=%p, buffer=%p, size=%d, mspinContext=%p){} ERROR: Connection broken -> abort sending data",
                __FUNCTION__, context, buffer, bufferSize, pMspinContext);
    }

    return NULL; //There is no way to indicate an error. This means that errors get ignored!
}

MSPIN_ERROR mspin_conn_init(void)
{
    MSPIN_ERROR ret = MSPIN_SUCCESS;
    //Initialize gCommCountMutex
    if(0 != pthread_mutex_init(&gCommCountMutex, NULL))
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s() FATAL ERROR: Failed to init gCommCountMutex", __FUNCTION__);
        ret = MSPIN_ERROR_GENERAL;
    }
    return ret;
}

MSPIN_ERROR mspin_conn_startUDPListener(S32 port, MSPIN_UDPMessageReceived mesgReceivedCB, void *pUDPListenerContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_CLIENT
    MSPIN_UNUSED(mesgReceivedCB);
    MSPIN_UNUSED(pUDPListenerContext);

    mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Not supported", __FUNCTION__, port);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_startUDPListener(port, mesgReceivedCB, pUDPListenerContext);
#endif //MSPIN_DISABLE_TCP_IP_CLIENT
}

MSPIN_ERROR mspin_conn_stopUDPListener(void)
{
#ifdef MSPIN_DISABLE_TCP_IP_CLIENT
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_stopUDPListener();
#endif //MSPIN_DISABLE_TCP_IP_CLIENT
}

MSPIN_ERROR mspin_conn_connectTCPServer(mspin_context_t* pContext, const U8* hostname, unsigned int port)
{
#ifdef MSPIN_DISABLE_TCP_IP_CLIENT
    MSPIN_UNUSED(pContext);
    MSPIN_UNUSED(hostname);
    MSPIN_UNUSED(port);

    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);

    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_connectServer(pContext, hostname, port);
#endif //MSPIN_DISABLE_TCP_IP_CLIENT
}

MSPIN_ERROR mspin_conn_disconnectTCPServer(mspin_context_t* pContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_CLIENT
    MSPIN_UNUSED(pContext);

    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_disconnectServer(pContext);
#endif //MSPIN_DISABLE_TCP_IP_CLIENT
}

MSPIN_ERROR mspin_conn_startTCPListener(S32 port, MSPIN_OnAcceptIncomingConnection acceptCB,
        MSPIN_OnConnectionClosed closedCB, void* callbackContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_startListener(port, NULL, acceptCB, closedCB, callbackContext);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_stopTCPListener(void)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_stopListener(FALSE);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_startTLSListener(S32 port, MSPIN_TLS_CONFIGURATION_t* pTLSConfig,
        MSPIN_OnAcceptIncomingConnection acceptCB, MSPIN_OnConnectionClosed closedCB, void* callbackContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
#ifdef MSPIN_DISABLE_TLS_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TLS not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_startListener(port, pTLSConfig, acceptCB, closedCB, callbackContext);
#endif //MSPIN_DISABLE_TLS_SERVER
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_stopTLSListener(void)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
#ifdef MSPIN_DISABLE_TLS_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TLS not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_stopListener(TRUE);
#endif //MSPIN_DISABLE_TLS_SERVER
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_closeTCPConnection(S32 connectionID)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(connectionID);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    in_addr_t ipAddr = mspin_tcp_getIPAddr(connectionID);
    MSPIN_ERROR result = mspin_tcp_closeConnection(connectionID);

    //When closed, issue connection closed callback. Otherwise only return error code
    if (MSPIN_SUCCESS == result)
    {
        mspin_tcp_signalConnectionClosed(connectionID, ipAddr, MSPIN_TCP_CONNECTION_REJECTED_BY_USER);
    }

    return result;
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_getMACAddress(S32 connectionID, U8* macAddress)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(connectionID);
    MSPIN_UNUSED(macAddress);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_getStoredMacAddress(connectionID, (char*)macAddress);

#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_startUDPBroadcasting(S32 udpPort, const U8* interface,
        MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd, void* onUDPBroadcastEndContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_udp_startUDPBroadcasting(udpPort, interface, messageParameters, timeout, onUDPBroadcastEnd, onUDPBroadcastEndContext);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_stopUDPBroadcasting(void)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_udp_stopUDPBroadcasting();
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_startUDPUnicasting(S32 udpPort, const U8* interface,
        const U8* destIPAddr, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters,
        U32 timeout, MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd,
        void* onUDPBroadcastEndContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_udp_startUDPUnicasting(udpPort, interface, (const char*)destIPAddr, messageParameters, timeout, onUDPBroadcastEnd, onUDPBroadcastEndContext);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_stopUDPUnicasting(void)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_udp_stopUDPUnicasting();
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_connectTCPClient(mspin_context_t* pContext, S32 connectionID)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(pContext);
    MSPIN_UNUSED(connectionID);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_connectClient(pContext, connectionID);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_disconnectTCPClient(mspin_context_t* pContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(pContext);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_disconnectClient(pContext);
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_connectTLSClient(mspin_context_t* pContext, S32 connectionID)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(connectionID);
    MSPIN_UNUSED(pContext);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TCP/IP not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
#ifdef MSPIN_DISABLE_TLS_SERVER
    MSPIN_UNUSED(connectionID);
    MSPIN_UNUSED(pContext);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TLS not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tls_connectClient(pContext, connectionID);
#endif //MSPIN_DISABLE_TLS_SERVER
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

MSPIN_ERROR mspin_conn_disconnectTLSClient(mspin_context_t* pContext)
{
#ifdef MSPIN_DISABLE_TCP_IP_SERVER
    MSPIN_UNUSED(pContext);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TCP/IP not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
#ifdef MSPIN_DISABLE_TLS_SERVER
    MSPIN_UNUSED(pContext);
    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: TLS not supported", __FUNCTION__);
    return MSPIN_ERROR_NOT_SUPPORTED;
#else
    return mspin_tcp_disconnectClient(pContext); //It's the same function as for TCP
#endif //MSPIN_DISABLE_TLS_SERVER
#endif //MSPIN_DISABLE_TCP_IP_SERVER
}

void mspin_conn_incrementCommCount(mspin_context_t* pContext)
{
    if (pContext)
    {
        (void)pthread_mutex_lock(&gCommCountMutex);
        ++(pContext->commCount);
        (void)pthread_mutex_unlock(&gCommCountMutex);
    }
}

void mspin_conn_decrementCommCount(mspin_context_t* pContext)
{
    if (pContext)
    {
        (void)pthread_mutex_lock(&gCommCountMutex);
        --(pContext->commCount);
        (void)pthread_mutex_unlock(&gCommCountMutex);
    }
}

S32 mspin_conn_getCommCount(mspin_context_t* pContext)
{
    return pContext ? pContext->commCount : 0;
}



#else //MSPINUSETCP

#define __USE_GNU
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>


mspin_connectionHandle_t* mspin_conn_connectAOAP(U32 vendorId, U32 productId, const char* serial)
{


    // connect
    mspin_log_printLn(eMspinVerbosityInfo, "%s(%#x,%#x,%s)",
            __FUNCTION__, vendorId, productId, serial);

    int tcpSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if ( tcpSocket == -1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: connection socket failed",
                __FUNCTION__);
        return NULL;
    }

//    struct sockaddr_in tcpAddr;
//    memset((void *)&tcpAddr, 0, sizeof(struct sockaddr_in));
//    tcpAddr.sin_family = AF_INET;
//    tcpAddr.sin_port = htons(serverPort);
//    tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

    struct sockaddr_in tcpServerAddr;
    memset((void *)&tcpServerAddr, 0, sizeof(struct sockaddr_in));
    tcpServerAddr.sin_family = AF_INET;
    tcpServerAddr.sin_port = htons(1234);
    tcpServerAddr.sin_addr.s_addr = inet_addr("127.0.0.1");

    int conRet = connect(tcpSocket, (struct sockaddr*)(&tcpServerAddr), sizeof(struct sockaddr) );
    if (conRet == -1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() no connection to server", __FUNCTION__);
        return NULL;
    }
    mspin_log_printLn(eMspinVerbosityInfo, "%s() connection to TCP server established, %d.",
            __FUNCTION__, tcpSocket);



    connectionHandle = MSPIN_createConnectionHandle();
    // keep the handle until connection gets disconnected
    if (connectionHandle)
    {
        connectionHandle->gAccessoryId = gAccessoryId;
        connectionHandle->deviceId = tcpSocket;
        connectionHandle->result = 0;
    }
    return connectionHandle;
}


size_t mspin_conn_receive(U8* buffer, size_t bufferSize, void* connectionHandle)
{
    size_t ret = 0;
    if (connectionHandle)
    {
        ret = recv(((mspin_connectionHandle_t*)connectionHandle)->deviceId, (void*)buffer, bufferSize, 0);

        mspin_log_printLn(eMspinVerbosityInfo, "%s() %d bytes received", __FUNCTION__, ret);
    }

    return ret;
}


void* mspin_conn_send(U8* buffer, size_t bufferSize, void* connectionHandle)
{
    if (connectionHandle)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() %d bytes on %d",
                __FUNCTION__, bufferSize, ((mspin_connectionHandle_t*)connectionHandle)->deviceId);
        send(((mspin_connectionHandle_t*)connectionHandle)->deviceId, (void*)buffer, bufferSize, 0);
        //mspin_dbgPrintLine(eMspinVerbosityVerbose, "MSPIN_Send %d bytes done.", bufferSize);

    }
    return NULL;
}
#endif //MSPINUSETCP
